package gov.va.med.mhv.integration.phr.dao;

import gov.va.med.mhv.integration.phr.transfer.Extract;
import gov.va.med.mhv.integration.phr.transfer.ExtractLog;
import gov.va.med.mhv.integration.phr.transfer.FacilityExtractStatus;
import gov.va.med.mhv.integration.phr.transfer.FacilityInfo;
import gov.va.med.mhv.integration.phr.transfer.PatientIdentifier;
import gov.va.med.mhv.integration.util.DateUtil;
import gov.va.med.mhv.integration.xss.esapi.EsapiXssOutputEncoderUtil;

import java.sql.Connection;
import java.sql.Date;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * DAO for logging and retrieving extract transfer statistics.
 * User: Rob Murtha
 
 */
public class TransferDAOImpl extends DaoSupportImpl implements TransferDAO {

    private String statusTable="V_PHR_EXTRACT_STATUS";

    /* (non-Javadoc)
	 * @see gov.va.med.mhv.integration.phr.dao.TransferDAO#setStatusTable(java.lang.String)
	 */
    @Override
	public void setStatusTable(String statusTable) {
        this.statusTable = statusTable;
    }

   /* (non-Javadoc)
 * @see gov.va.med.mhv.integration.phr.dao.TransferDAO#findAllFacilities()
 */
@Override
public Hashtable<String, FacilityInfo> findAllFacilities() {
       String sql = "select station_number, name from facility_info";
       Connection connection = null;
       PreparedStatement ps = null;
       ResultSet rs = null;
       Hashtable<String, FacilityInfo> results= new Hashtable<String, FacilityInfo>();

       try {
           connection=getConnection();
           ps = connection.prepareStatement(sql);
           rs = ps.executeQuery();
           while(rs.next()) {
               FacilityInfo value = new FacilityInfo();
               value.setName(rs.getString("name"));
               value.setStationNumber(rs.getString("station_number"));
               results.put(value.getStationNumber(), value);
           }
           rs.close();

       } catch (SQLException e) {
           throw new RuntimeException(e);
       } 
       catch (Exception ee) {
           throw new RuntimeException(ee);
       } 
       finally {
           cleanup(connection, rs,ps);
       }
       return results;
   }


    /* (non-Javadoc)
	 * @see gov.va.med.mhv.integration.phr.dao.TransferDAO#findStationNumbersByIcn(java.lang.String)
	 */
    @Override
	public Set<String> findStationNumbersByIcn(String icn) {
        String sql = "select station_number from phr_patient_info where icn = ? order by phr_facility_control_id desc";
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;
        Set<String> results= new HashSet<String>();

        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            ps.setString(1,icn);
            rs = ps.executeQuery();
            while(rs.next()) {
                results.add(rs.getString(1));
            }
            rs.close();

        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            cleanup(connection, rs,ps);
        }
        return results;
    }


    /* (non-Javadoc)
	 * @see gov.va.med.mhv.integration.phr.dao.TransferDAO#findEnabledExtracts()
	 */
    @Override
	public Map<String,Extract> findEnabledExtracts() {
        String sql = "select b.phr_extract_type_id, b.extract_name, a.phr_extract_enablement_type_id from phr_extract_control a join phr_extract_type b on ( a.phr_extract_type_id = b.phr_extract_type_id ) where a.phr_extract_enablement_type_id = 1";
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;
        Map<String,Extract> results= new HashMap<String,Extract>();

        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            rs = ps.executeQuery();
            while(rs.next()) {
                Extract value = new Extract(
                        rs.getLong(1),
                        rs.getString(2),
                        rs.getLong(3));
                results.put(value.getName(),value);                
            }
            rs.close();

        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            cleanup(connection, rs,ps);
        }
        return results;
    }

    /* (non-Javadoc)
	 * @see gov.va.med.mhv.integration.phr.dao.TransferDAO#save(gov.va.med.mhv.integration.phr.transfer.ExtractLog)
	 */
    @Override
	public void save(ExtractLog log)  {
        String sql =
                "insert into phr_extract_log "+
                "   (phr_extract_log_id,oplock,modified_date," +
                "    created_date,operation_name,status,icn,station_number,extract_type,start_date,end_date,record_count,retry_count, message_id)" +
                "values (HIBERNATE_SEQUENCE.nextval,0,systimestamp," +
                        "?,?,?,?,?,?,?,?,?,?,?)";
        Connection connection = null;
        PreparedStatement ps = null;
        
        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            int field=1;
            ps.setTimestamp(field++, DateUtil.dateToTimestamp(log.getCreatedDate()));
            ps.setString(field++, log.getOperationName());
            ps.setString(field++, log.getStatus());
            ps.setString(field++,log.getIcn());
            ps.setString(field++,log.getStationNumber());
            ps.setString(field++,log.getExtractType());
            ps.setTimestamp(field++, DateUtil.dateToTimestamp(log.getStartDate()));
            ps.setTimestamp(field++, DateUtil.dateToTimestamp(log.getEndDate()));
            ps.setLong(field++,log.getRecordCount());
            ps.setLong(field++,log.getRetryCount());
            ps.setLong(field++,log.getMessageId());

            ps.execute();

        } catch (SQLException e) {
            throw new RuntimeException(e);         
        } finally {
            cleanup(connection, ps);
        }
    }

    /* (non-Javadoc)
	 * @see gov.va.med.mhv.integration.phr.dao.TransferDAO#findEnabledExtractsByIcn(java.lang.String)
	 */
    @Override
	public List<FacilityExtractStatus> findEnabledExtractsByIcn(String icn) {
        String sql = "SELECT \n" +
                "    PHR_PATIENT_INFO.ICN, \n" +
                "    PHR_EXTRACT_TYPE.EXTRACT_NAME as EXTRACT_TYPE, \n" +
                "    PHR_PATIENT_INFO.STATION_NUMBER \n" +
                "FROM \n" +
                "    PHR_EXTRACT_CONTROL \n" +
                "INNER JOIN \n" +
                "    PHR_EXTRACT_TYPE \n" +
                "    ON \n" +
                "    (\n" +
                "        PHR_EXTRACT_CONTROL.PHR_EXTRACT_TYPE_ID = PHR_EXTRACT_TYPE.PHR_EXTRACT_TYPE_ID\n" +
                "    )\n" +
                "    , PHR_PATIENT_INFO \n" +
                "WHERE \n" +
                "    PHR_EXTRACT_CONTROL.PHR_EXTRACT_ENABLEMENT_TYPE_ID = 1 \n"+
                "    AND PHR_PATIENT_INFO.ICN                           = ? ";

        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;
        List<FacilityExtractStatus> result = new ArrayList<FacilityExtractStatus>();

        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            ps.setString(1,icn);
            rs = ps.executeQuery();
            while(rs.next()) {
                FacilityExtractStatus fes = new FacilityExtractStatus();
                fes.setIcn(encodeForXss(rs.getString("ICN")));
                fes.setStationNumber(encodeForXss(rs.getString("STATION_NUMBER")));
                fes.setExtract(encodeForXss(rs.getString("EXTRACT_TYPE")));
                result.add(fes);
            }

        } catch (SQLException e) {
            throw new RuntimeException(e);
        } 
        catch (Exception ee) {
            throw new RuntimeException(ee);
        } 
        finally {
            cleanup(connection,rs,ps);
        }
        return result;

    }
    

    public boolean isFieldTester(String icn, String accessRoleName) {
        String sql = "SELECT AR.NAME " +
					 "FROM ACCESS_ROLE AR, " +
					 "     USER_PROFILE_ACCESS_ROLE UPAR, " +
					 "     PATIENT P " +
					 "WHERE P.ICN = ? " +
					 "AND AR.NAME = ? " +
					 "AND UPAR.ACCESS_ROLE_ID  = AR.ACCESS_ROLE_ID " +
					 "AND UPAR.USER_PROFILE_ID = P.USER_PROFILE_USER_PROFILE_ID";
	    Connection connection = null;
	    PreparedStatement ps = null;
	    ResultSet rs = null;
	    boolean result = false;
	    
	    try {
	        connection=getConnection();
	        ps = connection.prepareStatement(sql);
	        ps.setString(1, icn);
	        ps.setString(2, accessRoleName);
	        rs = ps.executeQuery();            
	        if( rs.next() ) {
	        	//one row existed
	        	result = true;
	        }
	    } catch (SQLException e) {
	        throw new RuntimeException(e);
	    } finally {
	        cleanup(connection,rs,ps);
	    }
	    return result;
	}

    
	 /**
     * select LAST_COMPLETED_DATE from V_PHR_EXTRACT_STATUS where icn = '1012662547V377167' and extract_type = 'ChemistryHematology';
     */
    public Date findCompletedExtractByIcnAndMessageName(String icn, String messageName) {
        String sql = "select last_completed_date from "+statusTable+" where icn = ? and extract_type = ?";
        Connection connection = null;
        Date result = null;
        PreparedStatement ps = null;
        ResultSet rs = null;
        
        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            ps.setString(1,icn);
            ps.setString(2,messageName);
            rs = ps.executeQuery();            
            while(rs.next()) {
                result = rs.getDate("LAST_COMPLETED_DATE");
                break;
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            cleanup(connection,rs,ps);
        }
        return result;
    }
    
    /**
     * encodes/escapes database field to protect against Cross Site Scripting (XSS) Fortify flagged issues
     * 
     * @param output
     * @return
     * @throws Exception
     */
    private String encodeForXss(String output) throws Exception{
		return EsapiXssOutputEncoderUtil.encodeOutputForHtml(output);
	}

	@Override
	public List<PatientIdentifier> findFacilityPatientIds(PatientIdentifier pid) {
		
		List<PatientIdentifier> pids = null;
		
		if(pid.isIcn()) {
			pids = doIcnBasedFacilityPatientIdsLookup(pid);
		}
		else {
			pids = doLocalIdentifierBasedFacilityPatientIdsLookup(pid);
		}
		
		// filter out the identifiers that don't have the required fields
		return pids.stream().filter(
				p -> (!p.isIcn() && p.getIdentifier() != null && p.getAssigningAuthority() != null))
				.collect(Collectors.toList());
		
	}
	
	private List<PatientIdentifier> doIcnBasedFacilityPatientIdsLookup(PatientIdentifier pid) {
		String sql = "select identifier, type_code, name, assigning_authority from Patient p, Facility f "
				+ "where p.patient_id = f.patient_id and icn = ? and p.last_facilities_update > sysdate - 1";
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;
        List<PatientIdentifier> pids = new ArrayList<>();
        
        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            ps.setString(1,pid.getIdentifier());
            rs = ps.executeQuery();            
            while(rs.next()) {
                pids.add(new PatientIdentifier(rs.getString("identifier"), rs.getString("type_code"), rs.getString("name"), rs.getString("assigning_authority")));
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            cleanup(connection,rs,ps);
        }
        return pids;
	}
	
	private List<PatientIdentifier> doLocalIdentifierBasedFacilityPatientIdsLookup(PatientIdentifier pid) {
		String sql = "select identifier, type_code, name, assigning_authority from Patient p, Facility f"
				+ " where p.patient_id = f.patient_id and p.patient_id = (select distinct patient_id from Facility where name = ?"
				+ " and type_code = ? and identifier = ? and assigning_authority = ?) and p.last_facilities_update > sysdate - 1";
        Connection connection = null;
        PreparedStatement ps = null;
        ResultSet rs = null;
        List<PatientIdentifier> pids = new ArrayList<>();
        
        try {
            connection=getConnection();
            ps = connection.prepareStatement(sql);
            ps.setString(1,pid.getAssigningFacility());
            ps.setString(2, pid.getTypeCode());
            ps.setString(3, pid.getIdentifier());
            ps.setString(4, pid.getAssigningAuthority());
            rs = ps.executeQuery();            
            while(rs.next()) {
                pids.add(new PatientIdentifier(rs.getString("identifier"), rs.getString("type_code"), rs.getString("name"), rs.getString("assigning_authority")));
            }
        } catch (SQLException e) {
            throw new RuntimeException(e);
        } finally {
            cleanup(connection,rs,ps);
        }
        return pids;
	}

}
